function spring_mass_running
%% Tabula Rasa

clear all
close all
clc

%% Parameters

l0 = 1;         % uncompressed spring length [m]
m = 80;         % mass [kg]
k = 23000;		% spring stiffness [N/m]
g = 9.81;		% acceleration due to gravity [m/s^2]
alpha0 = 70;	% landing angle of leg [deg]

alpha0 = alpha0*pi/180;

%% Initial Conditions

x = 0;
y = 1;
x_dot = 5;
y_dot = 0;

x1 = -l0*cos(alpha0);  % horizontal distance between mass and foot contact at start of stance phase

%% State variable
q = [x y x_dot y_dot]'; % initial conditions

traj = [];

%% Transition Conditions

foot_impact = odeset('Events', @touch_down);    % event condition for landing
push_off = odeset('Events', @take_off);         % event condition for take off
peak = odeset('Events', @apex);                 % event contition for apex

%% Simulation
for i = 1:11
    [~,q] = ode45(@EoM_air, [0,5], q,foot_impact);  % integrate flight phase from apex
    
    traj = [traj; q];                               % store trajectory
    q = [x1 traj(end,2:4)]';                        % set initial conditions for stance phase, set origin to foot contact
    
    [~,q] = ode45(@EoM_ground, [0,5], q,push_off);  % integrate stance phase
    q(:,1) = q(:,1) + traj(end,1) + abs(x1);        % correct trajectory to include x displacement
    traj = [traj; q];                               % store trajectory
    q = traj(end,:);                                % set initial conditions for flight phase
    [~,q] = ode45(@EoM_air, [0,5], q,peak);         % integrate flight phase to apex
    traj = [traj; q];                               % store trajectory
    q = traj(end,:)';                               % set initial conditions for next step
end

%%

plot(traj(:,1),traj(:,2))
title('Trajectory of 10 steps for spring mass running')
xlabel('x (m)')
ylabel('y (m)')

%% Output Figure - uncomment to print a .eps file.
% print spring_mass_running_traj.eps -depsc2

%% Equation of Motion for Flight Phase
function output = EoM_air(~,q)
    output(1,1) = q(3);
    output(2,1) = q(4);
    output(3,1) = 0;
	output(4,1) = -g;
end
%% Equation of Motion for Stance Phase

% equations are in cartesian coordinates, makes switching between flight and
% stance phase easier, taken from 'The Spring-Mass Model for Running and
% Hopping' by R. Blickhan though you can derive it yourself easily enough

function output = EoM_ground(~,q)
    omega = sqrt(k/m);
	output(1,1) = q(3);
	output(2,1) = q(4);
	output(3,1) = q(1)*omega^2*(l0/sqrt(q(1)^2 + q(2)^2) - 1);
	output(4,1) = q(2)*omega^2*(l0/sqrt(q(1)^2 + q(2)^2) - 1) - g;
end
%% Conditions for step completion
% event for apex, y_dot = 0
function [impact,terminate,direction] = apex(~,q)
    impact = q(4);
	terminate = 1;
	direction = -1;
end
%% Conditions for take off
% event for take-off, spring length equals natural length l0^2 = x^2 + y^2
function [impact,terminate,direction] = take_off(~,q)
    impact = sqrt(q(1)^2 + q(2)^2) - l0;
	terminate = 1;
	direction = 1;
end
%% Conditions for touch down
% Condition for touch down given touch down angle alpha
function [impact,terminate,direction] = touch_down(~,q)
	impact = q(2) - l0*sin(alpha0);
	terminate = 1;
	direction = -1;
end
end